﻿/* 
 * Project:    SerialPort Terminal
 * Company:    Coad .NET, http://coad.net
 * Author:     Noah Coad, http://coad.net/noah
 * Created:    March 2005
 * 
 * Notes:      This was created to demonstrate how to use the SerialPort control for
 *             communicating with your PC's Serial RS-232 COM Port
 * 
 *             It is for educational purposes only and not sanctified for industrial use. :)
 *             Written to support the blog post article at: http://msmvps.com/blogs/coad/archive/2005/03/23/39466.aspx
 * 
 *             Search for "comport" to see how I'm using the SerialPort control.
 */

#region Namespace Inclusions

//#define USE_BPAC // Comment this line out if bpac is not needed
#if USE_BPAC     
using bpac;      //for optional Brother printer QL-820NWB
#endif

using System;
using System.Linq;
using System.Text;
using System.Drawing;
using System.IO.Ports;
using System.Windows.Forms;
using System.ComponentModel;

using SerialPortTerminal.Properties;
using System.Threading;
using System.IO;
using System.Net.Http;
using System.Net.Http.Headers;
using System.Timers;
using static System.Runtime.InteropServices.JavaScript.JSType;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using System.Diagnostics;
using static System.Windows.Forms.VisualStyles.VisualStyleElement;
using System.Text.Json;
using System.Security.Policy;

#endregion

namespace SerialPortTerminal
{

    #region Public Enumerations
    public enum DataMode { Text, Hex }
  public enum LogMsgType { Incoming, Outgoing, Normal, Warning, Error };
    #endregion

    public partial class frmTerminal : Form
    {

        #region Local Variables
        // The main control for communicating through the RS-232 port
        private SerialPort comport = new();

        // Various colors for logging info
        private Color[] LogMsgTypeColor = { Color.Blue, Color.Green, Color.Black, Color.Orange, Color.Red };

        // Temp holder for whether a key was pressed
        private bool KeyHandled = false;

        private Settings settings = Settings.Default;
        private string[] previousPorts;
        private PictureBox imageDisplay;
        private byte[] buffer = new byte[2560000]; // 2.5 MB buffer
        private static StringBuilder _dataBuffer = new();
        private readonly System.Timers.Timer _dataTimer;
        int bufferPosition = 0;
        string printData = "";
        string printPath = "";

        #endregion

        #region Constructor
        public frmTerminal()
        {
            // Load user settings
            settings.Reload();

            // Build the form
            InitializeComponent();
            this.StartPosition = FormStartPosition.CenterScreen;

            // Restore the users settings
            InitializeControlValues();

            // Enable/disable controls based on the current state
            EnableControls();

            // When data is recieved through the port, call this method
            comport.DataReceived += new SerialDataReceivedEventHandler(Port_DataReceived);
            comport.PinChanged += new SerialPinChangedEventHandler(comport_PinChanged);
            _dataTimer = new System.Timers.Timer(200);
            _dataTimer.Elapsed += new ElapsedEventHandler(OnDataTimerElapsed);
            Stopwatch stopwatch = new();
        }

        void comport_PinChanged(object sender, SerialPinChangedEventArgs e)
        {
            // Show the state of the pins
            UpdatePinState();
        }

        private void UpdatePinState()
        {
            this.Invoke(new ThreadStart(() =>
            {
                // Show the state of the pins
                chkCD.Checked = comport.CDHolding;
                chkCTS.Checked = comport.CtsHolding;
                chkDSR.Checked = comport.DsrHolding;
            }));
        }
        #endregion

        #region Local Methods

        /// <summary> Save the user's settings. </summary>
        private void SaveSettings()
        {
            settings.BaudRate = int.Parse(cmbBaudRate.Text);
            settings.DataBits = int.Parse(cmbDataBits.Text);
            //settings.DataMode = CurrentDataMode;
            settings.Parity = (Parity)Enum.Parse(typeof(Parity), cmbParity.Text);
            //settings.StopBits = (StopBits)Enum.Parse(typeof(StopBits), cmbStopBits.Text);
            settings.PortName = cmbPortName.Text;
            //settings.ClearOnOpen = chkClearOnOpen.Checked;
            //settings.ClearWithDTR = chkClearWithDTR.Checked;

            settings.Save();
        }

        /// <summary> Populate the form's controls with default settings. </summary>
        private void InitializeControlValues()
        {
            cmbParity.Items.Clear(); cmbParity.Items.AddRange(Enum.GetNames(typeof(Parity)));
            cmbStopBits.Items.Clear(); cmbStopBits.Items.AddRange(Enum.GetNames(typeof(StopBits)));

            cmbParity.Text = settings.Parity.ToString();
            //cmbStopBits.Text = settings.StopBits.ToString();
            cmbDataBits.Text = settings.DataBits.ToString();
            cmbParity.Text = settings.Parity.ToString();
            cmbBaudRate.Text = settings.BaudRate.ToString();
            //CurrentDataMode = settings.DataMode;

            PopulatePorts();

            //chkClearOnOpen.Checked = settings.ClearOnOpen;
            //chkClearWithDTR.Checked = settings.ClearWithDTR;

            // If it is still avalible, select the last com port used
            if (cmbPortName.Items.Contains(settings.PortName)) cmbPortName.Text = settings.PortName;
            else if (cmbPortName.Items.Count > 0) cmbPortName.SelectedIndex = cmbPortName.Items.Count - 1;
            else
            {
                MessageBox.Show(this, "There are no COM Ports detected on this computer.\nPlease install a COM Port and restart this app.", "No COM Ports Installed", MessageBoxButtons.OK, MessageBoxIcon.Error);
                this.Close();
            }
        }

        /// <summary> Enable/disable controls based on the app's current state. </summary>
        private void EnableControls()
        {
            // Enable/disable controls based on whether the port is open or not
            gbPortSettings.Enabled = !comport.IsOpen;
            txtSendData.Enabled = btnSend.Enabled = comport.IsOpen;
            //chkDTR.Enabled = chkRTS.Enabled = comport.IsOpen;

            if (comport.IsOpen) btnOpenPort.Text = "&Close Port";
            else btnOpenPort.Text = "&Open Port";
        }

        /// <summary> Send the user's data currently entered in the 'send' box.</summary>
        private void SendData()
        {
            /*if (CurrentDataMode == DataMode.Text)
            {
              // Send the user's text straight out the port
              comport.Write(txtSendData.Text);

              // Show in the terminal window the user's text
              Log(LogMsgType.Outgoing, txtSendData.Text + "\n");
            }
            else
            {*/
            try
            {
                // Convert the user's string of hex digits (ex: B4 CA E2) to a byte array
                byte[] data = HexStringToByteArray(txtSendData.Text);

                // Send the binary data out the port
                comport.Write(data, 0, data.Length);

                // Show the hex digits on in the terminal window
                Log(LogMsgType.Outgoing, ByteArrayToHexString(data) + "\n");
            }
            catch (FormatException)
            {
                // Inform the user if the hex string was not properly formatted
                Log(LogMsgType.Error, "Not properly formatted hex string: " + txtSendData.Text + "\n");
            }
            //}
            txtSendData.SelectAll();
        }

        /// <summary> Log data to the terminal window. </summary>
        /// <param name="msgtype"> The type of message to be written. </param>
        /// <param name="msg"> The string containing the message to be shown. </param>
        private void Log(LogMsgType msgtype, string msg)
        {
            rtfTerminal.Invoke(new EventHandler(delegate
            {
                rtfTerminal.SelectedText = string.Empty;
                rtfTerminal.SelectionFont = new Font(rtfTerminal.SelectionFont, FontStyle.Bold);
                rtfTerminal.SelectionColor = LogMsgTypeColor[(int)msgtype];
                rtfTerminal.AppendText(msg);
                rtfTerminal.ScrollToCaret();
            }));
        }

        /// <summary> Convert a string of hex digits (ex: E4 CA B2) to a byte array. </summary>
        /// <param name="s"> The string containing the hex digits (with or without spaces). </param>
        /// <returns> Returns an array of bytes. </returns>
        private byte[] HexStringToByteArray(string s)
        {
            s = s.Replace(" ", "");
            byte[] buffer = new byte[s.Length / 2];
            for (int i = 0; i < s.Length; i += 2)
                buffer[i / 2] = (byte)Convert.ToByte(s.Substring(i, 2), 16);
            return buffer;
        }

        /// <summary> Converts an array of bytes into a formatted string of hex digits (ex: E4 CA B2)</summary>
        /// <param name="data"> The array of bytes to be translated into a string of hex digits. </param>
        /// <returns> Returns a well formatted string of hex digits with spacing. </returns>
        private string ByteArrayToHexString(byte[] data)
        {
            StringBuilder sb = new StringBuilder(data.Length * 3);
            foreach (byte b in data)
                sb.Append(Convert.ToString(b, 16).PadLeft(2, '0').PadRight(3, ' '));
            return sb.ToString().ToUpper();
        }
        #endregion

        #region Local Properties
        private DataMode CurrentDataMode
        {
            get
            {
                if (rbHex.Checked) return DataMode.Hex;
                else return DataMode.Text;
            }
            set
            {
                if (value == DataMode.Text) rbText.Checked = true;
                else rbHex.Checked = true;
            }
        }
        #endregion

        #region Event Handlers

        private void frmTerminal_Shown(object sender, EventArgs e)
        {
            Log(LogMsgType.Normal, System.String.Format("Application Started at {0}\n", DateTime.Now));
        }
        private void frmTerminal_FormClosing(object sender, FormClosingEventArgs e)
        {
            if (e.CloseReason == CloseReason.UserClosing)
            {
                // Display a message box asking if the user wants to close the form
                DialogResult result = MessageBox.Show("Do you confirm closing the application?", "Close Application", MessageBoxButtons.YesNo, MessageBoxIcon.Question);
                if (result == DialogResult.No)
                {
                    // Cancel the form closing if the user clicks 'No'
                    e.Cancel = true;
                }
                else
                {
                    // If the port is open, close it.
                    if (comport != null && comport.IsOpen)
                    {
                        comport.DiscardInBuffer();
                        comport.DiscardOutBuffer();
                        comport.Close();
                    }
                    // The form is closing, save the user's preferences
                    SaveSettings();
                    Thread.Sleep(500);
                }
            }
        }

        private void RbText_CheckedChanged(object sender, EventArgs e)
        { if (rbText.Checked) CurrentDataMode = DataMode.Text; }

        private void RbHex_CheckedChanged(object sender, EventArgs e)
        { if (rbHex.Checked) CurrentDataMode = DataMode.Hex; }

        private void CmbBaudRate_Validating(object sender, CancelEventArgs e)
        { int x; e.Cancel = !int.TryParse(cmbBaudRate.Text, out x); }

        private void CmbDataBits_Validating(object sender, CancelEventArgs e)
        { int x; e.Cancel = !int.TryParse(cmbDataBits.Text, out x); }

        private void BtnOpenPort_Click(object sender, EventArgs e)
        {
            bool error = false;

            // If the port is open, close it.
            if (comport.IsOpen)
            {
                comport.Close();
                activityIndicator.BackColor = System.Drawing.Color.Red;
                PopulatePorts();
            }
            else
            {
                // Set the port's settings
                comport.BaudRate = int.Parse(cmbBaudRate.Text);
                comport.DataBits = 8; // int.Parse(cmbDataBits.Text);
                comport.StopBits = StopBits.One; // (StopBits)Enum.Parse(typeof(StopBits), cmbStopBits.Text);
                comport.Parity = (Parity)Enum.Parse(typeof(Parity), cmbParity.Text);
                comport.PortName = cmbPortName.Text;
                //comport.ReadTimeout = 1000;

                try
                {
                    // Open the port
                    comport.Open();
                    activityIndicator.BackColor = System.Drawing.Color.LightGreen;
                    UpdateLabel("COM Open");
                }
                catch (UnauthorizedAccessException) { error = true; }
                catch (IOException) { error = true; }
                catch (ArgumentException) { error = true; }

                if (error)
                {
                    MessageBox.Show(this, "Could not open the COM port.  Most likely it is already in use, has been removed, or is unavailable.", "COM Port Unavalible", MessageBoxButtons.OK, MessageBoxIcon.Stop);
                    activityIndicator.BackColor = System.Drawing.Color.Red;
                }
                else
                {
                    // Show the initial pin states
                    UpdatePinState();
                    chkDTR.Checked = comport.DtrEnable;
                    chkRTS.Checked = comport.RtsEnable;
                }
            }

            // Change the state of the form's controls
            EnableControls();

            // If the port is open, send focus to the send data box
            if (comport.IsOpen)
            {
                try
                {
                    // Clear the input buffer
                    comport.DiscardInBuffer();
                    comport.BaseStream.Flush();
                    // Clear the output buffer
                    comport.DiscardOutBuffer();

                    txtSendData.Focus();
                    if (chkClearOnOpen.Checked) ClearTerminal();
                }
                catch (Exception ex) { Console.WriteLine("Error: " + ex.Message); }
            }
        }
        private void BtnSend_Click(object sender, EventArgs e)
        { SendData(); }

        private void Port_DataReceived(object sender, SerialDataReceivedEventArgs e)
        {

            // If the com port has been closed, do nothing
            if (!comport.IsOpen) return;

            // This method will be called when there is data waiting in the port's buffer

            // Determain which mode (string or binary) the user is in
            if (CurrentDataMode == DataMode.Text)
            {
                // Read all the data waiting in the buffer
                //var sp = (SerialPort)sender;
                //string incomingData = sp.ReadExisting();
                //_dataBuffer.Append(incomingData);

                //_dataTimer.Stop();
                //_dataTimer.Start();

                //string data = comport.ReadExisting();
                //if (data.Length >= 10)
                //{
                //  if (data.Substring(0, 11) == "FaceDected\r")
                //  {

                //  }

                //}
                // Display the text to the user in the terminal
                //Log(LogMsgType.Incoming, data);
            }
            else
            {

                try
                {
                    if (comport.BytesToRead > 0)
                    {
                        int bytesRead = comport.Read(buffer, bufferPosition, buffer.Length - bufferPosition);
                        bufferPosition += bytesRead;
                        // Start the timer and wait to see if there are more data coming to the port
                        _dataTimer.Stop();
                        _dataTimer.Start();
                        activityIndicator.BackColor = System.Drawing.Color.Green;
                    }
                }
                catch (Exception ex)
                {
                    MessageBox.Show($"Failed to read data: {ex.Message}");
                }
            }
        }
        private void OnDataTimerElapsed(object sender, ElapsedEventArgs e)
        {
            _dataTimer.Stop();
            // Timer is expired and the captured data ready to process
            ProcessData(bufferPosition);
            bufferPosition = 0;
        }
        private async void ProcessData(int dataSize)
        {

            string dataHeader = System.Text.Encoding.ASCII.GetString(buffer, 0, Math.Min(dataSize, 20));
            // Check for linear barcode (less than 100 characters)
            if (dataSize < 100)
            {
                // if the data is DateOnly "]Jj", means Face detected, application will send ACK charecter to continue
                if (dataHeader.StartsWith("]Jj"))
                    comport.Write(new byte[] { 0x06 }, 0, 1);  // send ACK charecter to contine
                // Display data
                string textData = System.Text.Encoding.UTF8.GetString(buffer, 0, Math.Min(dataSize, 99));
                Log(LogMsgType.Incoming, "Barcode: " + textData + "\n");
            }
            // Check for two dimensional barcode (between 100 and 1000 characters)
            else if (dataSize <= 1000)
            {
                string textData = System.Text.Encoding.UTF8.GetString(buffer, 0, Math.Min(dataSize, dataSize));
                printData = textData; // Save this data for optional parse and printing via lable printer
                int length = textData.Length;

                // Define the file path
                string filePath = "output-PDF417.txt";
                // Write the string to the file
                File.WriteAllText(filePath, textData);

                rtfTerminal.Invoke(new Action(() =>
                {
                    rtfTerminal.Clear();
                }));
                Log(LogMsgType.Incoming, "2D Barcode: " + textData + "\n");
            }
            // Check for JPG image with header starting with "FaceDetected"
            else if (dataHeader.StartsWith("FaceDetected"))
            {
                int headerLength = "FaceDetected".Length;
                // 
                int imageStartIndex = headerLength + 1;
                int imageDataSize = dataSize - imageStartIndex - headerLength; // image size subtract from prefix (FaceDetected [CR] [LF]) and suffix (FaceDetected)

                // 
                byte[] imageData = new byte[imageDataSize];
                byte[] imageDataE = new byte[imageDataSize];
                Array.Copy(buffer, imageStartIndex, imageData, 0, imageDataSize);
                long timestamp = DateTime.Now.Ticks;
                string imagePathOriginal = $"{timestamp}-O.jpg";
                string imagePathEnhanced = $"{timestamp}-E.jpg";
                string imagePathJSON = $"{timestamp}-J.json";

                UpdateLabel("FaceDetected");

                //Save the original image into a file
                if (SavecheckBox.Checked)
                {
                    File.WriteAllBytes(imagePathOriginal, imageData);  // Write the original image into a file
                    printPath = imagePathOriginal;
                }
                // Display the original image
                Invoke(new Action(() =>
                {
                    try
                    {
                        using (MemoryStream ms = new MemoryStream(imageData))
                        {
                            // 
                            pictureBox1.Image = System.Drawing.Image.FromStream(ms);
                        }
                    }
                    catch (Exception ex)
                    {
                        MessageBox.Show($"Failed to display image: {ex.Message}");
                    }
                }));

                // If AI call is checked then send data to the AI server
                if (OCRcheckBox.Checked || AIcheckBox.Checked)
                {
                    //free demo conversion with DEMO watermark, no hash required
                    // curl -X POST -F "file=@a.jpg" "https://eightscans.com/upload_new.php?hash=1234567" - o a.json

                    string uploadUrl = "";
                    if (OCRcheckBox.Checked)
                        // This section is for OCR call for P30 scanner enabled for this functionality.
                        return;
                    else if (AIcheckBox.Checked)
                        uploadUrl = "https://eightscans.com/upload_new.php?hash=dadfafec13529c5935e0c854dd1ee39d";

                    string outputFile = imagePathJSON;

                    using var httpClient = new HttpClient();
                    using var form = new MultipartFormDataContent();
                    var byteArrayContent = new ByteArrayContent(imageData);

                    byteArrayContent.Headers.ContentType = MediaTypeHeaderValue.Parse("image/jpeg");

                    form.Add(byteArrayContent, "file", "image.jpg"); // "image.jpg" is a placeholder filename for the upload

                    HttpResponseMessage response = await httpClient.PostAsync(uploadUrl, form);
                    response.EnsureSuccessStatusCode();

                    string result = await response.Content.ReadAsStringAsync();

                    UpdateLabel("Got JASON");
                    if (SavecheckBox.Checked)
                    {
                        await System.IO.File.WriteAllTextAsync(outputFile, result);    // save the JSON file
                        Console.WriteLine("Upload complete. Response saved to " + outputFile);
                    }

                    try
                    {
                        string jsonContent = result;
                        using JsonDocument doc = JsonDocument.Parse(jsonContent);
                        JsonElement root = doc.RootElement;

                        //rtfTerminal.Clear();

                        foreach (JsonProperty prop in root.EnumerateObject())
                        {
                            string key = prop.Name;
                            JsonElement value = prop.Value;

                            if (key.ToLower() == "image")
                            {
                                // Try to load and display the image
                                try
                                {
                                    string base64Image = value.GetString();
                                    byte[] imageBytes = Convert.FromBase64String(base64Image);
                                    using MemoryStream ms = new MemoryStream(imageBytes);
                                    using Image tempImg = Image.FromStream(ms);
                                    pictureBox2.Invoke(new Action(() =>
                                    {
                                        pictureBox2.Image?.Dispose();
                                        pictureBox2.Image = new Bitmap(tempImg);
                                    }));
                                    if (SavecheckBox.Checked)
                                    {
                                        // Save to file
                                        string outputPath = imagePathEnhanced;
                                        using (Bitmap toSave = new Bitmap(tempImg))
                                        {
                                            toSave.Save(outputPath, System.Drawing.Imaging.ImageFormat.Jpeg);
                                            printPath = imagePathEnhanced;
                                        }
                                    }

                                }
                                catch
                                {
                                    rtfTerminal.Invoke(new Action(() =>
                                    {
                                        rtfTerminal.AppendText($"{key}: [Invalid or Unreadable Image]\n");
                                    }));
                                }
                            }
                            else if (value.ValueKind == JsonValueKind.Object)
                            {
                                rtfTerminal.Invoke(new Action(() =>
                                {
                                    rtfTerminal.AppendText($"{key}:\n");
                                }));

                                foreach (JsonProperty subProp in value.EnumerateObject())
                                {
                                    rtfTerminal.Invoke(new Action(() =>
                                    {
                                        if (subProp.Value.ValueKind == JsonValueKind.Array &&
                                            subProp.Value.GetArrayLength() == 2 &&
                                            subProp.Value[1].ValueKind == JsonValueKind.Number)
                                        {
                                            // Format CARD_ID values with confidence %
                                            string actualValue = subProp.Value[0].ToString();
                                            double confidence = subProp.Value[1].GetDouble();
                                            string confidenceFormatted = $"{confidence:0.0}%";

                                            rtfTerminal.AppendText($"    • {subProp.Name}: ");
                                            rtfTerminal.SelectionFont = new Font(rtfTerminal.Font, FontStyle.Bold);
                                            rtfTerminal.AppendText(actualValue);
                                            rtfTerminal.SelectionFont = new Font(rtfTerminal.Font, FontStyle.Regular);
                                            rtfTerminal.AppendText($"  (C: {confidenceFormatted})\n");
                                        }
                                        else if (key == "statistic" && subProp.Value.ValueKind == JsonValueKind.Number)
                                        {
                                            // Format statistic values to x.xxx
                                            double val = subProp.Value.GetDouble();
                                            string formatted = $"{val:0.000}";

                                            rtfTerminal.AppendText($"    • {subProp.Name}: ");
                                            rtfTerminal.SelectionFont = new Font(rtfTerminal.Font, FontStyle.Bold);
                                            rtfTerminal.AppendText(formatted);
                                            rtfTerminal.SelectionFont = new Font(rtfTerminal.Font, FontStyle.Regular);
                                            rtfTerminal.AppendText("\n");
                                        }
                                        else
                                        {
                                            // Fallback for other object content
                                            rtfTerminal.AppendText($"    • {subProp.Name}: ");
                                            rtfTerminal.SelectionFont = new Font(rtfTerminal.Font, FontStyle.Bold);
                                            rtfTerminal.AppendText(subProp.Value.ToString());
                                            rtfTerminal.SelectionFont = new Font(rtfTerminal.Font, FontStyle.Regular);
                                            rtfTerminal.AppendText("\n");
                                        }
                                    }));
                                }
                            }
                            else
                            {
                                // Display key-value data
                                rtfTerminal.Invoke(new Action(() =>
                                {
                                    rtfTerminal.AppendText($"{key}: {value}\n");
                                }));
                            }
                        }
                    }
                    catch (Exception ex)
                    {
                        MessageBox.Show($"Error loading image: {ex.Message}");
                    }
                }
            }
            else
            {
                Log(LogMsgType.Error, "Data does not match any category." + "\n");
            }
            activityIndicator.BackColor = System.Drawing.Color.LightGreen;
            UpdateLabel("Done");
        }
        
        private void txtSendData_KeyDown(object sender, KeyEventArgs e)
        {
            // If the user presses [ENTER], send the data now
            if (KeyHandled = e.KeyCode == Keys.Enter) { e.Handled = true; SendData(); }
        }
        private void txtSendData_KeyPress(object sender, KeyPressEventArgs e)
        { e.Handled = KeyHandled; }
        #endregion

        private void chkDTR_CheckedChanged(object sender, EventArgs e)
        {
            comport.DtrEnable = chkDTR.Checked;
            if (chkDTR.Checked && chkClearWithDTR.Checked) ClearTerminal();
        }

        private void chkRTS_CheckedChanged(object sender, EventArgs e)
        {
            comport.RtsEnable = chkRTS.Checked;
        }

        private void btnClear_Click(object sender, EventArgs e)
        {
            ClearTerminal();
            pictureBox1.Image = null;
        }

        private void ClearTerminal()
        {
            rtfTerminal.Clear();
        }

        private void PopulatePorts()
        {
            // Determain if the list of com port names has changed since last checked
            string[] ports = SerialPort.GetPortNames().Distinct().ToArray();
            // If there was an update, then update the control showing the user the list of port names
            if (previousPorts == null || !ArePortListsEqual(previousPorts, ports))
            {
                previousPorts = ports;
                cmbPortName.Items.Clear();
                foreach (string port in ports)
                {
                    cmbPortName.Items.Add(port);
                }
                if (cmbPortName.Items.Count > 0)
                {
                    cmbPortName.SelectedIndex = 0;
                }
            }
        }
        private bool ArePortListsEqual(string[] list1, string[] list2)
        {
            if (list1.Length != list2.Length)
                return false;
            Array.Sort(list1);
            Array.Sort(list2);
            for (int i = 0; i < list1.Length; i++)
            {
                if (list1[i] != list2[i])
                    return false;
            }
            return true;
        }

        private void PrintButton_Click(object sender, EventArgs e)
        {
            //  Uncomment This line on top #define USE_BPAC to active print function
#if USE_BPAC
        //this is for optional Brother printer QL-820NWB
        try
        {
            string templatePath = Environment.CurrentDirectory + "\\Visitor-01.lbx";
            string parserData;

            bpac.Document doc = new bpac.Document();

            if (doc.Open(templatePath))
            {
                doc.SetPrinter("Brother QL-820NWB", true);
                if (printData.Length > 0)
                {
                    DLParser parser = new DLParser(printData);
                    parserData = parser.DCS() + " " + parser.DAC();// + " " + parser.DAD();
                    doc.GetObject("visitor-name").Text = parserData; // "John Doe";
                    rtfTerminal.Text = parserData;
                }
                //MessageBox.Show("parser DCS" + parserData);
                if (printPath.Length > 0)
                {
                    doc.GetObject("visitor-image").SetData(0, printPath, 4);
                    //doc.GetObject("field2").Text = "...";
                }
                doc.StartPrint("", PrintOptionConstants.bpoDefault);
                doc.PrintOut(1, PrintOptionConstants.bpoDefault);
                doc.EndPrint();
                doc.Close();
            }
            else
            {
                MessageBox.Show("Open() Error: " + doc.ErrorCode);
            }
        }
        catch
        {
            MessageBox.Show(this, "Error Print", "Error Print", MessageBoxButtons.OK, MessageBoxIcon.Error);
        }
#else
            MessageBox.Show(this, "Refer to the source code for an optional print function to enhance your code's output.", "Print", MessageBoxButtons.OK);
#endif

        }

        private void label3_Click(object sender, EventArgs e)
        {

        }

        private void frmTerminal_Resize(object sender, EventArgs e)
        {
            int newWidth = this.Width / 3; // 1/3 of the form width
            int maxHeight = this.Height - (this.Height / 3); // Maximum allowed height

            // Maintain fixed aspect ratio of 1.285
            int newHeight = (int)(newWidth * 1.285);
            newHeight = Math.Min(newHeight, maxHeight); // Ensure height does not exceed limit

            pictureBox1.Width = newWidth;
            pictureBox1.Height = newHeight;
            pictureBox1.Location = new Point(this.Width / 3, pictureBox1.Location.Y);

            pictureBox2.Width = newWidth;
            pictureBox2.Height = newHeight;
            pictureBox2.Location = new Point((this.Width / 3) * 2, pictureBox2.Location.Y);

            // Adjust TextBox width and position
            rtfTerminal.Width = newWidth;
            rtfTerminal.Height = newHeight;



        }

        private void label5_Click(object sender, EventArgs e)
        {

        }

        private void AIcheckBox_CheckedChanged(object sender, EventArgs e)
        {
            // optional for the units with OCR option enabled
            //OCRcheckBox.Checked = !AIcheckBox.Checked;
        }

        private void OCRcheckBox_CheckedChanged(object sender, EventArgs e)
        {
            // optional for the units with OCR option enabled
            //AIcheckBox.Checked = !OCRcheckBox.Checked;
        }
        private void UpdateLabel(string StatusText)
        {
            if (labelStatus.InvokeRequired)
            {
                labelStatus.Invoke(new Action(() =>
                {
                    labelStatus.Text = StatusText;
                }));
            }
            else
            {
                labelStatus.Text = StatusText;
            }
        }
    }
}